const express = require("express");
const router = express.Router();
const mongoose = require("mongoose");
const asyncHandler = require("express-async-handler");
const Genre = require("../model/Genre");
const { uploadImage } = require("../helper/upload");
const authValidator = require("../middleware/authValidator");
const { isAdmin } = require("../middleware/bloggerAuth");
const moderationLogger = require("../middleware/moderationLogger");

// Helper: Generate slug from name
const generateSlug = (name) => {
  return name
    .toLowerCase()
    .trim()
    .replace(/[^\w\s-]/g, "")
    .replace(/\s+/g, "-")
    .replace(/-+/g, "-");
};

// CREATE GENRE
router.post(
  "/create",
  uploadImage.single("image"), 
  authValidator(),
  isAdmin,
  asyncHandler(async (req, res) => {
    const { name, description, content, isActive } = req.body;

    // Debugging line (optional, helps verify request body)
    console.log("Incoming genre body:", req.body);

    // Validation
    if (!name || !name.trim()) {
      res.status(400);
      throw new Error("Genre name is required");
    }

    const trimmedName = name.trim();
    const slug = generateSlug(trimmedName);

    // Check if genre already exists
    const existing = await Genre.findOne({
      $or: [{ name: trimmedName }, { slug }],
      isDeleted: false,
    });
    if (existing) {
      res.status(409);
      throw new Error("Genre already exists");
    }

    // Handle uploaded image (like register route)
    let imageUrl = null;
    if (req.file) {
      imageUrl = `/uploads/images/${req.file.filename}`;
    }

    // Create genre
    const genre = await Genre.create({
      name: trimmedName,
      slug,
      description: description?.trim() || null,
      content: content?.trim() || null,
      image: imageUrl,
      isActive: isActive !== undefined ? isActive : true,
      ...(req.user && { "meta.createdBy": req.user._id }),
    });

    res.status(201).json({
      success: true,
      message: "Genre created successfully",
      data: genre,
    });
  })
);

// READ All Genres (with filters)
router.get(
  "/fetch",
  asyncHandler(async (req, res) => {
    const { isActive, search, page = 1, limit = 10 } = req.query;

    // Build query
    const query = { isDeleted: false };

    if (isActive !== undefined) {
      query.isActive = isActive === "true";
    }

    if (search) {
      query.$or = [
        { name: { $regex: search, $options: "i" } },
        { description: { $regex: search, $options: "i" } },
      ];
    }

    // Pagination
    const skip = (parseInt(page) - 1) * parseInt(limit);
    const total = await Genre.countDocuments(query);

    const genres = await Genre.find(query)
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit))
      .populate("meta.createdBy", "name email")
      .select("-__v");

    res.status(200).json({
      success: true,
      count: genres.length,
      total,
      page: parseInt(page),
      pages: Math.ceil(total / parseInt(limit)),
      data: genres,
    });
  })
);

// READ One Genre (by ID or slug)
router.get(
  "/fetch/:id",
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    let genre;

    // Check if it's a valid ObjectId
    if (mongoose.Types.ObjectId.isValid(id)) {
      genre = await Genre.findOne({ _id: id, isDeleted: false })
        .populate("meta.createdBy", "name email")
        .select("-__v");
    } else {
      // Treat as slug
      genre = await Genre.findOne({ slug: id, isDeleted: false })
        .populate("meta.createdBy", "name email")
        .select("-__v");
    }

    if (!genre) {
      res.status(404);
      throw new Error("Genre not found");
    }

    res.status(200).json({
      success: true,
      data: genre,
    });
  })
);

// UPDATE Genre
router.put(
  "/edit/:id",
  uploadImage.single("image"), // same multer logic as register/create
  authValidator(),
  isAdmin,
  asyncHandler(async (req, res) => {
    const { id } = req.params;
    const { name, description, content, isActive } = req.body;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Genre ID");
    }

    // Prepare update object
    const updateData = {};

    // ===== Handle Name + Slug =====
    if (name && name.trim()) {
      const trimmedName = name.trim();
      const slug = generateSlug(trimmedName);

      // Check for name or slug conflicts
      const existing = await Genre.findOne({
        $or: [{ name: trimmedName }, { slug }],
        _id: { $ne: id },
        isDeleted: false,
      });

      if (existing) {
        res.status(409);
        throw new Error("Genre name already exists");
      }

      updateData.name = trimmedName;
      updateData.slug = slug;
    }

    // ===== Description, Content, Active Flag =====
    if (description !== undefined) {
      updateData.description = description?.trim() || null;
    }

    if (content !== undefined) {
      updateData.content = content?.trim() || null;
    }

    if (isActive !== undefined) {
      updateData.isActive = isActive;
    }

    // ===== Image Upload =====
    if (req.file) {
      updateData.image = `/uploads/images/${req.file.filename}`;
    }

    // ===== Update Operation =====
    const genre = await Genre.findOneAndUpdate(
      { _id: id, isDeleted: false },
      updateData,
      { new: true, runValidators: true }
    )
      .populate("meta.createdBy", "name email")
      .select("-__v");

    if (!genre) {
      res.status(404);
      throw new Error("Genre not found");
    }

    res.status(200).json({
      success: true,
      message: "Genre updated successfully",
      data: genre,
    });
  })
);

// SOFT DELETE Genre
router.delete(
  "/delete/:id",
  authValidator(),
  isAdmin,
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Genre ID");
    }

    const genre = await Genre.findOneAndUpdate(
      { _id: id, isDeleted: false },
      { isDeleted: true, isActive: false },
      { new: true }
    );

    if (!genre) {
      res.status(404);
      throw new Error("Genre not found");
    }

    res.status(200).json({
      success: true,
      message: "Genre deleted successfully",
    });
  })
);

// HARD DELETE Genre (admin only - use with caution)
router.delete(
  "/:id/permanent",
  authValidator(),
  isAdmin,
  moderationLogger("DELETE_GENRE", "genre"),
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Genre ID");
    }

    const genre = await Genre.findByIdAndDelete(id);

    if (!genre) {
      res.status(404);
      throw new Error("Genre not found");
    }

    res.status(200).json({
      success: true,
      message: "Genre permanently deleted",
    });
  })
);

// RESTORE Soft-deleted Genre
router.patch(
  "/:id/restore",
  authValidator(),
  isAdmin,
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Genre ID");
    }

    const genre = await Genre.findOneAndUpdate(
      { _id: id, isDeleted: true },
      { isDeleted: false },
      { new: true }
    );

    if (!genre) {
      res.status(404);
      throw new Error("Genre not found or not deleted");
    }

    res.status(200).json({
      success: true,
      message: "Genre restored successfully",
      data: genre,
    });
  })
);

// TOGGLE Active Status
router.patch(
  "/:id/toggle-active",
  authValidator(),
  isAdmin,
  asyncHandler(async (req, res) => {
    const { id } = req.params;

    if (!mongoose.Types.ObjectId.isValid(id)) {
      res.status(400);
      throw new Error("Invalid Genre ID");
    }

    const genre = await Genre.findOne({ _id: id, isDeleted: false });

    if (!genre) {
      res.status(404);
      throw new Error("Genre not found");
    }

    genre.isActive = !genre.isActive;
    await genre.save();

    res.status(200).json({
      success: true,
      message: `Genre ${
        genre.isActive ? "activated" : "deactivated"
      } successfully`,
      data: genre,
    });
  })
);

module.exports = router;
