const jwt = require("jsonwebtoken");
const { JWT_SECRET } = require("../env.json");
const User = require("../model/user");

const authValidator = () => {
  return async (req, res, next) => {
    try {
      const cookieAuthtoken = req.cookies?.authtoken;
      if (!cookieAuthtoken) return next();

        const tokenUser = jwt.verify(cookieAuthtoken, JWT_SECRET);

      const dbUser = await User.findById(tokenUser.id).populate(
        "favouriteGenres"
      );
      if (!dbUser) return next();

      const {
        _id,
        firstName,
        lastName,
        username,
        email,
        phone,
        role,   
        shortBio,
        occupation,
        favouriteGenres,
        pfp,
      } = dbUser;

      const userPayload = {
        id: _id,
        firstName,
        lastName,
        username,
        email,
        phone,
        role,
        shortBio,
        occupation,
        favouriteGenres,
        pfp,
      };

      const authtoken = jwt.sign(userPayload, JWT_SECRET, { expiresIn: "1h" });

      // Base cookie options
      let cookieOptions = {
        httpOnly: true,
        secure: process.env.NODE_ENV === "production",
        sameSite: "lax",
        path: "/",
      };

      // For admin, set 1-hour expiry
      if (role === "admin") {
        cookieOptions.expires = new Date(Date.now() + 60 * 60 * 1000);
      }

      res.cookie("authtoken", authtoken, cookieOptions);

      req.user = userPayload;
      next();
    } catch (err) {
      if (err.name === "TokenExpiredError") {
        console.warn("JWT expired, clearing cookie...");
        res.clearCookie("authtoken", {
          httpOnly: true,
          secure: process.env.NODE_ENV === "production",
          sameSite: "lax",
          path: "/",
        });
      } else {
        console.error("AuthValidator Error:", err.message);
      }
      next();
    }
  };
};

module.exports = authValidator;
